(function(){
  const FX_ENDPOINT = "https://api.frankfurter.dev/v1/latest?base=EUR&symbols=USD,GBP,ZAR";
  function fmt4(x){ return (Math.round(x*10000)/10000).toFixed(4); }
  async function fetchRates(){
    const fallback = { date:"offline", base:"EUR", rates:{ USD:1.10, GBP:0.86, ZAR:20.50 }, ts:new Date() };
    try{
      const res = await fetch(FX_ENDPOINT, { cache:"no-store" });
      if(!res.ok) throw new Error("FX HTTP " + res.status);
      const data = await res.json();
      return { date:data.date, base:data.base, rates:data.rates, ts:new Date() };
    }catch(e){
      return fallback;
    }
  }
  function computeTiles(r){
    const EURUSD = Number(r.rates.USD);
    const EURGBP = Number(r.rates.GBP);
    const EURZAR = Number(r.rates.ZAR);
    const GBPUSD = EURUSD / EURGBP;
    const ZARUSD = EURUSD / EURZAR;
    return [
      { label:"EUR → USD", value: EURUSD, from:"EUR", fromCountry:"Europe" },
      { label:"GBP → USD", value: GBPUSD, from:"GBP", fromCountry:"United Kingdom" },
      { label:"ZAR → USD", value: ZARUSD, from:"ZAR", fromCountry:"South Africa" },
      { label:"USD → USD", value: 1.0, from:"USD", fromCountry:"United States" }
    ];
  }
  function renderFx(stripEl, r, onPick){
    if(!stripEl) return;
    stripEl.innerHTML = "";
    const stamp = r.date === "offline" ? "Offline fallback" : ("Date " + r.date);
    computeTiles(r).forEach(t=>{
      const el = document.createElement("div");
      el.className = "fxTile";
      el.setAttribute("role","button");
      el.tabIndex = 0;
      el.innerHTML =
        '<div class="fxTop">'+
          '<div class="fxPair">'+t.label+'</div>'+
          '<div class="fxLive">LIVE</div>'+
        '</div>'+
        '<div class="fxMid">'+
          '<div>'+
            '<div class="fxBig">1.00 '+t.from+'</div>'+
            '<div class="fxSub">= <strong>'+fmt4(t.value)+'</strong> USD</div>'+
          '</div>'+
          '<button class="fxArrow" aria-hidden="true">'+
            '<svg viewBox="0 0 24 24" fill="none">'+
              '<path d="M9 18l6-6-6-6" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>'+
            '</svg>'+
          '</button>'+
        '</div>'+
        '<div class="fxStamp">'+stamp+'</div>';
      el.addEventListener("click", ()=>{ if(onPick) onPick(t); });
      stripEl.appendChild(el);
    });
  }
  window.SendZimFX = { fetchRates, renderFx, fmt4, computeTiles };
})();
