<?php
declare(strict_types=1);

function app_config(): array {
  static $cfg = null;
  if ($cfg !== null) return $cfg;

  $cfgFile = __DIR__ . '/../config/config.php';
  if (!file_exists($cfgFile)) {
    $cfg = require __DIR__ . '/../config/config.sample.php';
    $cfg['__missing_config'] = true;
    return $cfg;
  }
  $cfg = require $cfgFile;
  return $cfg;
}

function db(): mysqli {
  static $db = null;
  if ($db !== null) return $db;

  $cfg = app_config();
  $dbcfg = $cfg['db'];
  $db = new mysqli($dbcfg['host'], $dbcfg['user'], $dbcfg['pass'], $dbcfg['name']);
  if ($db->connect_errno) {
    http_response_code(500);
    die("DB connection failed: " . $db->connect_error);
  }
  $db->set_charset($dbcfg['charset'] ?? 'utf8mb4');
  return $db;
}

function db_exec(string $sql, array $params = [], string $types = ''): mysqli_stmt {
  $stmt = db()->prepare($sql);
  if (!$stmt) { http_response_code(500); die("DB prepare failed"); }
  if ($params) {
    if ($types === '') {
      $types = '';
      foreach ($params as $p) $types .= is_int($p) ? 'i' : (is_float($p) ? 'd' : 's');
    }
    $stmt->bind_param($types, ...$params);
  }
  if (!$stmt->execute()) { http_response_code(500); die("DB execute failed"); }
  return $stmt;
}

function db_row(string $sql, array $params = []): ?array {
  $stmt = db_exec($sql, $params);
  $res = $stmt->get_result();
  $row = $res ? $res->fetch_assoc() : null;
  $stmt->close();
  return $row ?: null;
}

function db_all(string $sql, array $params = []): array {
  $stmt = db_exec($sql, $params);
  $res = $stmt->get_result();
  $rows = [];
  if ($res) { while ($r = $res->fetch_assoc()) $rows[] = $r; }
  $stmt->close();
  return $rows;
}


function db_has_column(string $table, string $column): bool {
  $cfg = app_config();
  $dbName = $cfg['db']['name'] ?? '';
  $row = db_row(
    "SELECT 1 AS ok FROM information_schema.COLUMNS WHERE TABLE_SCHEMA=? AND TABLE_NAME=? AND COLUMN_NAME=? LIMIT 1",
    [$dbName, $table, $column]
  );
  return (bool)$row;
}
