<?php
declare(strict_types=1);
require_once __DIR__ . '/tronscan.php';

/**
 * Official USD₮ (USDT) TRC20 contract address published by Tether:
 * https://tether.to/supported-protocols/  (TRC20 Token via Tron Blockchain).
 */
const SENDZIM_USDT_TRC20_CONTRACT = 'TR7NHqjeKQxGTCi8q8ZY4pL8otSzgjLj6t';

/**
 * Verify a TRC20 USDT payment by txid.
 *
 * Strategy:
 * - Prefer /api/token_trc20/transfers (requires TRONSCAN API key for reliable access).
 * - Fallback to /api/transaction-info (sometimes works without key, but may not include TRC20 transfer fields).
 *
 * @return array{ok:bool, status:string, message:string, amount:float|null, from:string|null, to:string|null, confirmed:bool|null}
 */
function verify_usdt_trc20_txid(string $txid, string $expected_to, float $min_amount_usdt, float $tolerance = 0.02): array {
  $txid = trim($txid);
  if ($txid === '') return ['ok'=>false,'status'=>'invalid','message'=>'Missing txid','amount'=>null,'from'=>null,'to'=>null,'confirmed'=>null];

  // 1) token_trc20/transfers (best)
  $res = tronscan_get('/api/token_trc20/transfers', [
    'limit' => 20,
    'start' => 0,
    'confirm' => 'true',
    'contract_address' => SENDZIM_USDT_TRC20_CONTRACT,
    'toAddress' => $expected_to,
  ]);

  if ($res['ok'] && isset($res['data']['token_transfers']) && is_array($res['data']['token_transfers'])) {
    foreach ($res['data']['token_transfers'] as $t) {
      $hash = (string)($t['transaction_id'] ?? $t['hash'] ?? '');
      if (strcasecmp($hash, $txid) !== 0) continue;

      $dec = (int)($t['decimals'] ?? $t['tokenInfo']['tokenDecimal'] ?? 6);
      $raw = $t['quant'] ?? $t['amount'] ?? $t['value'] ?? null;
      $amt = null;
      if (is_numeric($raw)) $amt = ((float)$raw) / pow(10, $dec);

      $from = (string)($t['from_address'] ?? $t['fromAddress'] ?? '');
      $to   = (string)($t['to_address']   ?? $t['toAddress']   ?? '');

      $confirmed = (bool)($t['confirmed'] ?? true);

      if ($to !== '' && strcasecmp($to, $expected_to) !== 0) {
        return ['ok'=>false,'status'=>'mismatch','message'=>'Tx found but sent to different address.','amount'=>$amt,'from'=>$from ?: null,'to'=>$to ?: null,'confirmed'=>$confirmed];
      }
      if ($amt !== null && $amt + $tolerance < $min_amount_usdt) {
        return ['ok'=>false,'status'=>'underpaid','message'=>'Tx found but amount is lower than expected.','amount'=>$amt,'from'=>$from ?: null,'to'=>$to ?: null,'confirmed'=>$confirmed];
      }
      return ['ok'=>true,'status'=>'verified','message'=>'Payment verified.','amount'=>$amt,'from'=>$from ?: null,'to'=>$to ?: null,'confirmed'=>$confirmed];
    }
    return ['ok'=>false,'status'=>'not_found','message'=>'Txid not found yet (may still be propagating).','amount'=>null,'from'=>null,'to'=>null,'confirmed'=>null];
  }

  // If unauthorized and no key configured, be explicit.
  if (!$res['ok'] && (int)$res['status'] === 401 && tronscan_api_key() === '') {
    return ['ok'=>false,'status'=>'needs_api_key','message'=>'Auto-verify needs a TRONSCAN API key (TRON-PRO-API-KEY). TXID saved; admin can verify manually.','amount'=>null,'from'=>null,'to'=>null,'confirmed'=>null];
  }

  // 2) Fallback: transaction-info
  $info = tronscan_get('/api/transaction-info', ['hash' => $txid]);
  if (!$info['ok'] || !isset($info['data']) || !is_array($info['data'])) {
    return ['ok'=>false,'status'=>'lookup_failed','message'=>'Could not verify right now. TXID saved; try again in a minute.','amount'=>null,'from'=>null,'to'=>null,'confirmed'=>null];
  }
  $d = $info['data'];
  $confirmed = (bool)($d['confirmed'] ?? false);

  // Attempt to extract TRC20 transfer fields (varies by tx type)
  $to = (string)($d['toAddress'] ?? $d['contractData']['to_address'] ?? $d['contractData']['toAddress'] ?? '');
  $from = (string)($d['ownerAddress'] ?? $d['contractData']['owner_address'] ?? '');
  $token = $d['contractData']['tokenInfo']['tokenId'] ?? $d['contractData']['tokenInfo']['tokenAddress'] ?? null;
  $token = is_string($token) ? $token : '';

  $dec = (int)($d['contractData']['tokenInfo']['tokenDecimal'] ?? 6);
  $raw = $d['contractData']['amount'] ?? $d['contractData']['value'] ?? null;
  $amt = null;
  if (is_numeric($raw)) $amt = ((float)$raw) / pow(10, $dec);

  if ($to !== '' && strcasecmp($to, $expected_to) !== 0) {
    return ['ok'=>false,'status'=>'mismatch','message'=>'Tx found but sent to different address.','amount'=>$amt,'from'=>$from ?: null,'to'=>$to ?: null,'confirmed'=>$confirmed];
  }
  if ($token !== '' && strcasecmp($token, SENDZIM_USDT_TRC20_CONTRACT) !== 0) {
    return ['ok'=>false,'status'=>'wrong_token','message'=>'Tx found but token is not USDT (TRC20).','amount'=>$amt,'from'=>$from ?: null,'to'=>$to ?: null,'confirmed'=>$confirmed];
  }
  if ($amt !== null && $amt + $tolerance < $min_amount_usdt) {
    return ['ok'=>false,'status'=>'underpaid','message'=>'Tx found but amount is lower than expected.','amount'=>$amt,'from'=>$from ?: null,'to'=>$to ?: null,'confirmed'=>$confirmed];
  }
  if (!$confirmed) {
    return ['ok'=>false,'status'=>'pending','message'=>'Tx found but not confirmed yet.','amount'=>$amt,'from'=>$from ?: null,'to'=>$to ?: null,'confirmed'=>false];
  }
  return ['ok'=>true,'status'=>'verified','message'=>'Payment verified.','amount'=>$amt,'from'=>$from ?: null,'to'=>$to ?: null,'confirmed'=>$confirmed];
}
