<?php
declare(strict_types=1);
require_once __DIR__ . '/db.php';

/**
 * TRONSCAN HTTP helper.
 * If config['tronscan']['api_key'] is set, we'll send header TRON-PRO-API-KEY.
 * Docs: https://docs.tronscan.org/ (API Keys).
 */
function tronscan_api_key(): string {
  $cfg = app_config();
  return (string)($cfg['tronscan']['api_key'] ?? '');
}

function tronscan_get(string $path, array $query = []): array {
  $base = 'https://apilist.tronscanapi.com';
  $url = rtrim($base, '/') . '/' . ltrim($path, '/');
  if (!empty($query)) {
    $url .= (strpos($url, '?') === false ? '?' : '&') . http_build_query($query);
  }

  $ch = curl_init($url);
  if ($ch === false) return ['ok'=>false,'status'=>0,'error'=>'curl_init_failed'];

  $headers = ['Accept: application/json'];
  $key = tronscan_api_key();
  if ($key !== '') $headers[] = 'TRON-PRO-API-KEY: ' . $key;

  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_TIMEOUT => 12,
    CURLOPT_CONNECTTIMEOUT => 6,
    CURLOPT_HTTPHEADER => $headers,
    CURLOPT_SSL_VERIFYPEER => true,
  ]);

  $body = curl_exec($ch);
  $status = (int)curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
  $err = curl_error($ch);
  curl_close($ch);

  if ($body === false) return ['ok'=>false,'status'=>$status,'error'=>$err ?: 'curl_exec_failed'];

  $json = json_decode($body, true);
  if (!is_array($json)) {
    return ['ok'=>false,'status'=>$status,'error'=>'invalid_json','raw'=>$body];
  }
  return ['ok'=>($status>=200 && $status<300), 'status'=>$status, 'data'=>$json];
}
